/**
 * \file sdc_perm.h
 *
 * \brief libSDC: Definition of functions etc. required by to manage permissions of
 * keys - <b>should not be included directly!</b>
 *
 * Application should not include this header directly!
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/


/**
 * \ingroup keymanagement
 * \defgroup keymanagement_perm Permission managment
 * Functions to manage permission used during key creation (see \ref keymanagement_keystore)
 */


#ifndef __SDC_LIB_PERM_H_
#define __SDC_LIB_PERM_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>

/* Definitions types and defaults */

/**
 * \ingroup keymanagement_perm
 * \brief Integer type to store access permissions
 *
 *  This must fit all permission bits in SDC_PERM_...
 *
 */
typedef uint16_t sdc_perm_bmsk_t;

/**
 * \ingroup keymanagement_perm
 * \defgroup keymanagement_perm_mask Permission masks
 * Defines to describe permissions of \ref sdc_perm_bmsk_t in \ref sdc_permissions_t
 */

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow data encryption
 */
#define SDC_PERM_ENCRYPT     0x0001

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow data decryption
 */
#define SDC_PERM_DECRYPT     0x0002

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow to generate tag of data
 */
#define SDC_PERM_SIGN        0x0004

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow to verify tag of data
 */
#define SDC_PERM_VERIFY      0x0008

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow data wrap (encryption + authentication)
 */
#define SDC_PERM_WRAP        0x0010

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow data unwrap (decryption + authentication)
 */
#define SDC_PERM_UNWRAP      0x0020

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow key wrap (encryption + authentication)
 */
#define SDC_PERM_WRAP_KEY    0x0040

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow key to be unwrap (decryption + authentication)
 */
#define SDC_PERM_UNWRAP_KEY  0x0080

/**
 * \ingroup keymanagement_perm_mask
 * \brief Allow to delete a key
 * Due to architectures using sdc_daemon this permission must only be
 * available for owner (i.e. \ref sdc_permissions_t perms_owner and \ref sdc_permissions_t perms_inherit_owner)
 */
#define SDC_PERM_DELETE      0x0100

/**
 * \ingroup keymanagement_perm_mask
 * \brief Mask to describe no operation is supported
 */

#define SDC_PERM_NONE        0x0000

/**
 * \ingroup keymanagement_perm_mask
 * \brief Mask to allow all data operations
 * Allow encrypt, decrypt, sign, verify, wrap and unwrap
 */
#define SDC_PERM_ALL_NORM_OP (SDC_PERM_ENCRYPT | SDC_PERM_DECRYPT | \
                              SDC_PERM_SIGN | SDC_PERM_VERIFY | \
                              SDC_PERM_WRAP | SDC_PERM_UNWRAP)

/**
 * \ingroup keymanagement_perm_mask
 * \brief Max permission an user can gain
 */
#define SDC_PERM_UID_MAX                (SDC_PERM_ALL_NORM_OP | SDC_PERM_DELETE | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Max permission a group can gain
 */
#define SDC_PERM_GID_MAX                (SDC_PERM_ALL_NORM_OP | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Max permission for others
 */
#define SDC_PERM_OTHERS_MAX             (SDC_PERM_ALL_NORM_OP | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Max inheritance mask for user
 * In case the user permissions are relevant when importing an protected key this
 * value limits the permission which can be set for the imported key
 */
#define SDC_PERM_INHERIT_UID_MAX        (SDC_PERM_ALL_NORM_OP | SDC_PERM_DELETE | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Max inheritance mask for group
 * In case the group permissions are relevant when importing an protected key this
 * value limits the permission which can be set for the imported key
 */
#define SDC_PERM_INHERIT_GID_MAX        (SDC_PERM_ALL_NORM_OP | SDC_PERM_DELETE | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Max inheritance mask for others
 * In case the others permissions are relevant when importing an protected key this
 * value limits the permission which can be set for the imported key
 */
#define SDC_PERM_INHERIT_OTHERS_MAX     (SDC_PERM_ALL_NORM_OP | SDC_PERM_DELETE | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)

/**
 * \ingroup keymanagement_perm_mask
 * \brief Default permission for user
 */
#define SDC_PERM_UID_DFLT               (SDC_PERM_ALL_NORM_OP | SDC_PERM_DELETE | SDC_PERM_WRAP_KEY | SDC_PERM_UNWRAP_KEY)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Default permission for group
 */
#define SDC_PERM_GID_DFLT               (SDC_PERM_ALL_NORM_OP)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Default permission for others
 */
#define SDC_PERM_OTHERS_DFLT            (SDC_PERM_NONE)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Default inheritance mask for user
 */
#define SDC_PERM_INHERIT_UID_DFLT       (SDC_PERM_INHERIT_UID_MAX)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Default inheritance mask for group
 */
#define SDC_PERM_INHERIT_GID_DFLT       (SDC_PERM_INHERIT_GID_MAX)
/**
 * \ingroup keymanagement_perm_mask
 * \brief Default inheritance mask for others
 */
#define SDC_PERM_INHERIT_OTHERS_DFLT    (SDC_PERM_NONE)

/**
 * \ingroup keymanagement_perm
 * \brief Type to specify the permission and group during key creation
 *
 * <b>The elements of this struct must not be accessed directly!</b>
 * In order to allocate, free, read or modify its elements the corresponding
 * functions need to be used.
 * See ::sdc_permissions_alloc, ::sdc_permissions_free ::sdc_set_default_permissions_and_gid ::sdc_set_default_permissions_current_gid
 */
typedef struct sdc_permissions sdc_permissions_t;

/* Functions */

/**
 * \ingroup keymanagement_perm
 * \brief Allocate permissions structure
 *
 * After allocation the content of the struct needs to be filled using functions
 * like ::sdc_set_default_permissions_and_gid, ::sdc_set_default_permissions_current_gid.
 * After usage the structure needs to be freed using ::sdc_permissions_free.
 *
 * \param[out] permissions a pointer to the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_alloc (sdc_permissions_t **permissions);

/**
 * \ingroup keymanagement_perm
 * \brief Free permissions structure
 *
 * Free the structure allocated by ::sdc_permissions_alloc.
 *
 * \param[in] permissions the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_free (sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_perm
 * \brief Set default permissions and explicitly configured group
 *
 * Note: uid is set to current effective uid
 *
 * Set default permissions and an explicitly configured group.
 *
 * \param[in] permissions the structure pointer
 * \param[in] gid group identifier of the desired group
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_set_default_permissions_and_gid (sdc_permissions_t *permissions, gid_t gid);

/**
 * \ingroup keymanagement_perm
 * \brief Set default permissions
 *
 * Note: uid is set to current effective uid
 *
 * In contrast to ::sdc_set_default_permissions_and_gid the group is configured implicitly.
 * This function will set the group to the current effective group of the calling process.
 *
 * \param[in] permissions the structure pointer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_set_default_permissions_current_gid (sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_perm
 * \brief set gid of permissions
 *
 * \param[in] permissions the structure pointer
 * \param[in] gid to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_set_gid (sdc_permissions_t *permissions, gid_t gid);

/**
 * \ingroup keymanagement_perm
 * \brief set uid of permissions
 *
 * Note: For keystore keys the UID needs to be always equal to the effective uid
 *
 * \param[in] permissions the structure pointer
 * \param[in] uid uid to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_set_uid (sdc_permissions_t *permissions, uid_t uid);

/**
 * \ingroup keymanagement_perm
 * \brief Get gid of permissions
 *
 * \param[in] permissions the structure pointer
 * \param[out] gid pointer to return the gid
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_get_gid (const sdc_permissions_t *permissions, gid_t *gid);

/**
 * \ingroup keymanagement_perm
 * \brief Get uid of permissions
 *
 * \param[in] permissions the structure pointer
 * \param[out] uid pointer to return the uid
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_get_uid (const sdc_permissions_t *permissions, gid_t *uid);

/**
 * \ingroup keymanagement_perm
 * \brief Parse a string containing access permissions
 *
 *  The syntax follows the syntax known from Unix and chmod.
 *
 * \<cmd_string\> ::= \<cmd\>(,\<cmd\>)*
 * \<cmd\>        ::= \<target\>\<operator\>\<flags\>
 * \<target\>     ::= (u|g|o|a|U|G|O|A)*
 * \<operator\>   ::= "+"|"-"|"="
 * \<flags\>      ::= (e|d|s|v|w|u|k|D)*
 *
 * The targets:
 * ============
 *   u - user/owner
 *   g - group
 *   o - others
 *   a - all, shortcut for ugo
 *   U - inheritance: user
 *   G - inheritance: group
 *   O - inheritance: others
 *   A - inheritance: all, shortcut for UGO
 *
 * The operators:
 * ==============
 *   + - Set the specified flags in the specified target(s)
 *   - - Clear the specified flags from the specified target(s)
 *   = - Set the specified flags in the specified target(s) and clear others
 *
 * The permission flags:
 * =====================
 *   e - encrypt
 *   d - decrypt
 *   s - sign
 *   v - verify
 *   w - wrap
 *   u - unwrap
 *   k - wrap key
 *   K - unwrap key
 *   D - delete the key
 *
 * Space characters are ignored.
 * If the target
 * \param[in] perms the structure pointer
 * \param[in] str - NULL terminated string which is to parse
 * \param[in] maxlen - Maximum possible length of string
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer or syntax error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_apply_string(sdc_permissions_t *perms,
                                                 const char *str,
                                                 size_t maxlen);

/**
 * \ingroup keymanagement_perm
 * \brief Pretty print permissions into a string
 *
 * \param[in] perms the structure pointer
 * \param[in] buffer pointer to a buffer to write into
 * \param[in] len size of the buffer
 *
 * The buffer size required is always 60 bytes. So len must be at least
 * 60 and is here for security reasons.
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - null pointer passed or buffer too small
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_write(const sdc_permissions_t *perms,
                                          char *buffer, size_t len);

/** Setters and getters **/


/**
 * \ingroup keymanagement_perm
 * \brief Set permission for owner.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_owner_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get permissions of owner.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_owner_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Set inheritance mask for owner.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_owner_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get inheritance mask of owner.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_owner_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Set permission for group.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_group_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get permissions of group.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_group_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Set inheritance mask for group.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_group_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get inheritance mask of group.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_group_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Set permission for others.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_others_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get permissions of others.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_others_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Set inheritance mask for others.
 *
 * \param[in] permissions the structure pointer
 * \param[in] bmsk permission bitmask to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_others_set(sdc_permissions_t *permissions, sdc_perm_bmsk_t bmsk);

/**
 * \ingroup keymanagement_perm
 * \brief Get inheritance mask of others.
 *
 * \param[in] permissions the structure pointer
 * \param[out] bmsk pointer to return the current permissions
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PERM_INVALID - one or multiple bits in bmsk exceed max allowed permissions
 * \return \ref SDC_INVALID_PARAMETER - null pointer
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_permissions_inherit_others_get(const sdc_permissions_t *permissions, sdc_perm_bmsk_t *bmsk);

#ifdef __cplusplus
}
#endif

#endif

